#!/bin/bash

# --- Load libraries
# check if the logging.sh file exists and is readable
lib_path="$HOME/.local/lib"
if [ -r "$lib_path/logging.sh" ]; then
    # Source the logging library
    source "$lib_path/logging.sh"
else
    # Define a simple fallback logging function
    log() {
        # Echo the last argument passed to the function
        echo "${@: -1}"
    }
fi

# --- Read variables

# define defaults in case nothing is passed
# @DOCS:
# - for when ANSIBLE_CONFIG variable might not be available yet; e.g. at setup
ansible_config_path="${HOME}/.ansible.cfg" # default value

# function to show usage
usage() {
  echo "Usage: $0 [-acp|--ansible-config-path config_path]"
  exit 1
}

# parse arguments
# @reminder: shift once for flag args, shift 2 (twice) for arg with value
while [[ "$#" -gt 0 ]]; do
  case $1 in
    -acp|--ansible-config-path)
        # for when ANSIBLE_CONFIG variable might not be available yet; e.g. at setup
        ansible_config_path="$2"
        # shift once for flag args, shift 2 (twice) for arg with value
        shift 2
        ;; 
    *) usage ;; # Show usage for unknown option
  esac
  shift # Move to next argument
done

log INFO "Checking if passed ansible config file located at ${ansible_config_path} exists..."
if [ ! -f "${ansible_config_path}" ]; then
    log ERROR "Provided ansible config path ${ansible_config_path} is not valid file. Abort."
    exit 1
fi
log INFO "Provided ansible-roles dir exists. However, it was not checked whether it is valid git repo with expected name ansible-roles."

# --- Repo setup
log INFO $BLUE "Setting up data_collector_lib repo for dev..."
script_dir=$(dirname "$(readlink -f "${BASH_SOURCE[0]}")")
ANSIBLE_CONFIG=$ansible_config_path ansible-playbook \
    -i "localhost," -c local \
    $script_dir/.setup/repo-setup.yaml
log INFO $BLUE "data_collector_lib repo set up for dev successful."
