import datetime
import json
import logging
import logging.config

from data_collector.eodhd.utils import (
    read_object_from_aws_s3_bucket,
    save_json_to_aws_s3_bucket,
    assume_role,
    wipe_credentials,
)
from data_collector.eodhd.tasks import (
    get_eodhd_exchanges_list,
    get_eodhd_total_tickers_list,
    select_tickers_to_download_funddata_for,
    get_eodhd_total_fundamental_ticker_data,
)


logger = logging.getLogger(__name__)


# vanilla task orchestration
# - for more advanced, try prefect ..
def main(app_config: dict):
    # read in config
    eodhd_token = app_config["eodhd_token"]
    daily_api_limit = app_config["daily_api_limit"]
    aws_role_arn = app_config["arn_data_collector_role_to_assume"]
    aws_s3_bucket_name = app_config["data_collector_s3_bucket_name"]
    data_collector_sts_aws_access_key_id = app_config[
        "data_collector_sts_aws_access_key_id"
    ]
    data_collector_sts_aws_secret_key_id = app_config[
        "data_collector_sts_aws_secret_key_id"
    ]

    # params
    today_utc = datetime.datetime.now(datetime.timezone.utc)
    today_utc_str = today_utc.strftime("%Y-%m-%d")

    # actions

    # assume role
    aws_sts_credentials = assume_role(
        aws_role_arn,
        data_collector_sts_aws_access_key_id,
        data_collector_sts_aws_secret_key_id,
    )

    # task 1
    s3_key = f"eodhd/exchanges_list/{today_utc_str}/UTC/obj.json"
    logger.info("Check if exchanges list fetched already, else call API.")
    exchanges_list = get_eodhd_exchanges_list(
        aws_sts_credentials=aws_sts_credentials,
        aws_s3_bucket_name=aws_s3_bucket_name,
        s3_key=s3_key,
        eodhd_token=eodhd_token,
    )
    logger.info(f"s3_key: {s3_key}; Saving the exchanges list...")
    task_1_success = save_json_to_aws_s3_bucket(
        json_to_save=json.dumps(exchanges_list),
        s3_key=s3_key,
        aws_sts_credentials=aws_sts_credentials,
        aws_s3_bucket_name=aws_s3_bucket_name,
    )
    logger.info(f"success: {task_1_success}; Task 1 done.")

    # task 2
    s3_key = f"eodhd/total_tickers/{today_utc_str}/UTC/obj.json"
    logger.info("Check if total tickers list fetched already, else call API.")
    total_tickers = get_eodhd_total_tickers_list(
        exchanges_list=exchanges_list,
        s3_key=s3_key,
        aws_s3_bucket_name=aws_s3_bucket_name,
        aws_sts_credentials=aws_sts_credentials,
        eodhd_token=eodhd_token,
    )
    logger.info(f"s3_key: {s3_key}; Saving the total_tickers list...")
    task_2_success = save_json_to_aws_s3_bucket(
        json_to_save=json.dumps(total_tickers),
        s3_key=s3_key,
        aws_sts_credentials=aws_sts_credentials,
        aws_s3_bucket_name=aws_s3_bucket_name,
    )
    logger.info(f"success: {task_2_success}; Task 2 done.")

    # task 3
    s3_key = f"eodhd/ticker_fundamental_data/{today_utc_str}/UTC/obj.json"

    # check if already written for the day
    # - if yes, come back tomorrow
    tickers_funddata_entry_s3_key = (
        f"eodhd/ticker_fundamental_data/{today_utc_str}/UTC/obj.json"
    )
    logger.info("Checking if have already written fundamental data for the day...")
    tickers_funddata_entry = read_object_from_aws_s3_bucket(
        aws_sts_credentials=aws_sts_credentials,
        aws_s3_bucket_name=aws_s3_bucket_name,
        s3_key=tickers_funddata_entry_s3_key,
    )
    if tickers_funddata_entry is not None:
        logger.info("Have already written fundamental data for today. Job done.")
        return

    logger.info("Select which tickers to download fundamental data for...")
    tickers_to_download = select_tickers_to_download_funddata_for(
        aws_sts_credentials=aws_sts_credentials, aws_s3_bucket_name=aws_s3_bucket_name
    )

    if len(tickers_to_download) == 0:
        logger.info(
            "If nothing left to download, append today as the next day whose tickers list we work to download fundamental data for..."
        )
        total_tickers_dam_s3_key = "eodhd/total_tickers_dam/obj.json"
        dam_json = read_object_from_aws_s3_bucket(
            aws_sts_credentials=aws_sts_credentials,
            aws_s3_bucket_name=aws_s3_bucket_name,
            s3_key=total_tickers_dam_s3_key,
        )
        dam_json = json.loads(dam_json)
        dam_json.append(today_utc_str)
        save_json_to_aws_s3_bucket(
            json_to_save=json.dumps([today_utc_str]),
            s3_key=s3_key,
            aws_sts_credentials=aws_sts_credentials,
            aws_s3_bucket_name=aws_s3_bucket_name,
        )
        logger.info(
            "After just made today as the DAM day, re-evaluate which tickers are left to download fundamental data for..."
        )
        tickers_to_download = select_tickers_to_download_funddata_for(
            aws_sts_credentials=aws_sts_credentials,
            aws_s3_bucket_name=aws_s3_bucket_name,
        )

    tickers_to_download = tickers_to_download[:daily_api_limit]
    logger.info(
        f"Decided which tickers we download fundamental data for today. Count: {len(tickers_to_download)}. Downloading..."
    )
    chunk_funddata = get_eodhd_total_fundamental_ticker_data(
        total_tickers=tickers_to_download,
        aws_sts_credentials=aws_sts_credentials,
        aws_s3_bucket_name=aws_s3_bucket_name,
        s3_key=s3_key,
        eodhd_token=eodhd_token,
    )
    logger.info(f"s3_key: {s3_key}; Saving the fundamental data chunk...")
    task_3_success = save_json_to_aws_s3_bucket(
        json_to_save=json.dumps(chunk_funddata),
        s3_key=s3_key,
        aws_sts_credentials=aws_sts_credentials,
        aws_s3_bucket_name=aws_s3_bucket_name,
    )
    logger.info(f"success: {task_3_success}; Task 3 done.")

    # wipe creds
    aws_sts_credentials = wipe_credentials(aws_sts_credentials)
    logger.info("Workflow completed.")
