import requests
import datetime
import json
import time
import logging
from typing import List

from data_collector.eodhd.utils import (
    read_object_from_aws_s3_bucket,
    save_json_to_aws_s3_bucket,
)


logger = logging.getLogger(__name__)


# task 1
# - get list of exchanges
# - save down the list


def get_eodhd_exchanges_list_from_eodhd_api(eodhd_token: str):
    logger.debug("Fetching exchange list from EODHD API...")
    api_endpoint = (
        f"https://eodhd.com/api/exchanges-list/?api_token={eodhd_token}&fmt=json"
    )
    response = requests.get(api_endpoint)
    return response.json()


def get_eodhd_exchanges_list(
    aws_sts_credentials: dict, aws_s3_bucket_name: str, s3_key: str, eodhd_token: str
):
    """
    Attept read from S3 and then default to API if needed.
    Equality between the S3 source and the API source was tested and it is reliable.
    """
    logger.debug("Attempting to check if exchanges list is fetch already for today...")
    from_s3 = read_object_from_aws_s3_bucket(
        aws_s3_bucket_name=aws_s3_bucket_name,
        s3_key=s3_key,
        aws_sts_credentials=aws_sts_credentials,
    )
    if from_s3 is not None:
        logger.debug(
            "Exchange list was found already saved for today. Returning cached."
        )
        return json.loads(from_s3)
    return get_eodhd_exchanges_list_from_eodhd_api(eodhd_token=eodhd_token)


# task 2
# - get union list of tickers from list of exchanges
# - save down the union list


def get_eodhd_tickers_per_exchange_from_eodhd_api(eodhd_token: str, exchange_code: str):
    logger.debug(
        f"Fetching tickers list from EODHD API for exchange={exchange_code}..."
    )
    api_endpoint = f"https://eodhd.com/api/exchange-symbol-list/{exchange_code}?api_token={eodhd_token}&fmt=json"
    response = requests.get(api_endpoint)
    return response.json()


def get_eodhd_total_tickers_list(
    exchanges_list: List,
    aws_sts_credentials: dict,
    aws_s3_bucket_name: str,
    s3_key: str,
    eodhd_token: str,
):
    """
    Attept read from S3 and then default to API if needed.
    Equality between the S3 source and the API source was tested and it is reliable.
    """
    logger.debug("Attempting to check if total tickers is fetch already for today...")
    from_s3 = read_object_from_aws_s3_bucket(
        aws_s3_bucket_name=aws_s3_bucket_name,
        s3_key=s3_key,
        aws_sts_credentials=aws_sts_credentials,
    )
    if from_s3 is not None:
        logger.debug(
            "Tickers list was found already saved for today. Returning cached."
        )
        return json.loads(from_s3)
    logger.debug("Tickers list needs to be fetched from EODHD API.")

    total_tickers = []
    logger.info(f"Exchanges list count: {len(exchanges_list)}")
    for i, exch in enumerate(exchanges_list):
        logger.debug(f'Current exchange {i}: {exch["Name"]}, {exch["Code"]}')
        curr_tickers = get_eodhd_tickers_per_exchange_from_eodhd_api(
            eodhd_token=eodhd_token, exchange_code=exch["Code"]
        )
        total_tickers += curr_tickers
    return total_tickers


# task 3
# - get ticker level fundamental data for entire union list
# - save down the union list


def get_eodhd_fundamental_data_per_ticker_from_eodhd_api(eodhd_token: str, ticker: str):
    # set up time interval logs
    current_time = time.time()
    if not hasattr(
        get_eodhd_fundamental_data_per_ticker_from_eodhd_api, "last_log_time"
    ):
        get_eodhd_fundamental_data_per_ticker_from_eodhd_api.last_log_time = (
            0  # Initialize last log time
        )

    # Log if more than 30 seconds have passed
    can_log = False
    if (
        current_time
        - get_eodhd_fundamental_data_per_ticker_from_eodhd_api.last_log_time
        > 30
    ):
        can_log = True
        get_eodhd_fundamental_data_per_ticker_from_eodhd_api.last_log_time = (
            current_time
        )

    if can_log:
        logger.debug(f"Fetching fundamental data from EODHD API for ticker={ticker}...")
    api_endpoint = (
        f"https://eodhd.com/api/fundamentals/{ticker}?api_token={eodhd_token}&fmt=json"
    )
    response = requests.get(api_endpoint)
    if response.status_code == 200:
        return response.json()
    return None


def select_tickers_to_download_funddata_for(
    aws_sts_credentials: dict, aws_s3_bucket_name: str
):
    """
    Motivation:
    - due to API limits bound, tickers fundamental data is downloaded over a few days
    - we store a 'download active mark' (DAM) for ticker fundamental data
    - that is the ticker's **list** (as per task 2) of which day is currently under download?
    - when done for day, diff the union of everything downloaded so far
      - decide whether done completely with the current DAM
    """
    today_utc = datetime.datetime.now(datetime.timezone.utc)
    today_utc_str = today_utc.strftime("%Y-%m-%d")

    # access DAM to fetch initial state
    s3_key = "eodhd/total_tickers_dam/obj.json"
    logger.info(
        "Fetching Download Active Mark (DAM) file, the state file for where the batch download for fundamental data was left..."
    )
    dam_json = read_object_from_aws_s3_bucket(
        aws_sts_credentials=aws_sts_credentials,
        aws_s3_bucket_name=aws_s3_bucket_name,
        s3_key=s3_key,
    )

    # if empty, fill in with [today] and reattempt
    if dam_json is None:
        logger.info(
            "Download Active Mark (DAM) file is absent. Filling with today and re-calling current function to re-evaluate..."
        )
        save_json_to_aws_s3_bucket(
            json_to_save=json.dumps([today_utc_str]),
            s3_key=s3_key,
            aws_sts_credentials=aws_sts_credentials,
            aws_s3_bucket_name=aws_s3_bucket_name,
        )
        select_tickers_to_download_funddata_for(
            aws_sts_credentials=aws_sts_credentials,
            aws_s3_bucket_name=aws_s3_bucket_name,
        )

    dam_json = json.loads(dam_json)
    tickers_list_resume_download_date = max(
        [datetime.datetime.strptime(d, "%Y-%m-%d") for d in dam_json]
    )
    logger.info(
        f'Download Active Mark (DAM) file says that we are working towards the fundamental data of the ticker list from {tickers_list_resume_download_date.strftime("%Y-%m-%d")}...'
    )
    tickers_list_resume_download_s3_key = f'eodhd/total_tickers/{tickers_list_resume_download_date.strftime("%Y-%m-%d")}/UTC/obj.json'

    logger.info(
        f'Reading total tickers list from {tickers_list_resume_download_date.strftime("%Y-%m-%d")}...'
    )
    dam_tickers_list = read_object_from_aws_s3_bucket(
        aws_sts_credentials=aws_sts_credentials,
        aws_s3_bucket_name=aws_s3_bucket_name,
        s3_key=tickers_list_resume_download_s3_key,
    )
    dam_tickers_list = json.loads(dam_tickers_list)

    logger.info(
        f"Computing which tickers we have downloaded data from so far of that active for download ticker list..."
    )
    dam_tickers_done_so_far = {}
    curr_date = tickers_list_resume_download_date
    while curr_date < datetime.datetime.strptime(today_utc_str, "%Y-%m-%d"):
        tickers_funddata_entry_s3_key = f'eodhd/ticker_fundamental_data/{curr_date.strftime("%Y-%m-%d")}/UTC/obj.json'
        tickers_funddata_entry = read_object_from_aws_s3_bucket(
            aws_sts_credentials=aws_sts_credentials,
            aws_s3_bucket_name=aws_s3_bucket_name,
            s3_key=tickers_funddata_entry_s3_key,
        )
        if tickers_funddata_entry is None:
            curr_date = curr_date + datetime.timedelta(days=1)
            continue
        tickers_funddata_entry = json.loads(tickers_funddata_entry)
        for tck in tickers_funddata_entry:
            dam_tickers_done_so_far[tck["Code"]] = True
        curr_date = curr_date + datetime.timedelta(days=1)

    logger.info(
        f"Performing the set difference between what is needed (total list) and what we have already..."
    )
    res = []
    for tck in dam_tickers_list:
        if not tck["Code"] in dam_tickers_done_so_far:
            res.append(tck)
    logger.info(f"Done. Returning tickers left to download fundamental data for...")
    return res


def get_eodhd_total_fundamental_ticker_data(
    total_tickers: List,
    aws_sts_credentials: dict,
    aws_s3_bucket_name: str,
    s3_key: str,
    eodhd_token: str,
):
    """
    Attept read from S3 and then default to API if needed.
    Equality between the S3 source and the API source was tested and it is reliable.
    """
    logger.debug(
        "Attempting to check if fundamental data is fetch already for today..."
    )
    from_s3 = read_object_from_aws_s3_bucket(
        aws_s3_bucket_name=aws_s3_bucket_name,
        s3_key=s3_key,
        aws_sts_credentials=aws_sts_credentials,
    )
    if from_s3 is not None:
        logger.debug("Fundamental data fetched for today. Returning cache...")
        logger.warning(
            "This could be a bug. The intended tickers list for fundamentals download and what is found need not be in sync... Think through."
        )
        return json.loads(from_s3)

    total_funddata = []
    logger.info(f"Tickers list count: {len(total_tickers)}")
    for i, tck in enumerate(total_tickers):
        if i % 100 == 0:
            logger.info(f'Current ticker {i}: {tck["Name"]}')
        curr_funddata = get_eodhd_fundamental_data_per_ticker_from_eodhd_api(
            eodhd_token=eodhd_token, ticker=tck["Code"]
        )
        if curr_funddata is not None:
            total_funddata.append(curr_funddata)
    return total_funddata
