import boto3
from botocore.exceptions import ClientError
import time
import logging
import uuid

logger = logging.getLogger(__name__)


def assume_role(
    role_arn,
    sts_aws_access_key_id: str,
    sts_aws_secret_key_id: str,
    session_base_name="eodhd-fdmtl-download",
):
    """Assume an AWS role and return temporary credentials."""
    logger.debug(f"role_arn: {role_arn}; Assuming aws iam role...")
    sts_client = boto3.client(
        "sts",
        aws_access_key_id=sts_aws_access_key_id,
        aws_secret_access_key=sts_aws_secret_key_id,
    )
    response = sts_client.assume_role(
        RoleArn=role_arn,
        RoleSessionName=f"{session_base_name}-{str(uuid.uuid4())[:20]}",
        DurationSeconds=900,
    )
    logger.debug("Done assuming aws iam role.")
    # Extract credentials from the assumed role response
    credentials = response["Credentials"]
    return {
        "AccessKeyId": credentials["AccessKeyId"],
        "SecretAccessKey": credentials["SecretAccessKey"],
        "SessionToken": credentials["SessionToken"],
    }


# for assume_role above
# - delete credentials of session early
# - might be needed or not depending on prefect flow paradigm also (still to learn)
# - remember that you can use minimal time of assume session = 900 seconds
def wipe_credentials(credentials):
    """Post-run task to wipe credentials by setting them to None."""
    logger.info("Wiping credentials...")
    credentials = None
    return credentials


def read_object_from_aws_s3_bucket(
    aws_s3_bucket_name: str, s3_key: str, aws_sts_credentials: dict
):
    """
    Fetch and return the content of an object. Return None if the object does not exist.
    """
    # set up time interval logs
    current_time = time.time()
    if not hasattr(read_object_from_aws_s3_bucket, "last_log_time"):
        read_object_from_aws_s3_bucket.last_log_time = 0  # Initialize last log time

    # Log if more than 30 seconds have passed
    can_log = False
    if current_time - read_object_from_aws_s3_bucket.last_log_time > 30:
        can_log = True
        read_object_from_aws_s3_bucket.last_log_time = current_time

    s3 = boto3.client(
        "s3",
        aws_access_key_id=aws_sts_credentials["AccessKeyId"],
        aws_secret_access_key=aws_sts_credentials["SecretAccessKey"],
        aws_session_token=aws_sts_credentials["SessionToken"],
    )

    try:
        response = s3.get_object(Bucket=aws_s3_bucket_name, Key=s3_key)
        res = response["Body"].read().decode("utf-8")
        if can_log:
            logger.debug(f"Object {s3_key} was found in bucket {aws_s3_bucket_name}.")
        return res
    except ClientError as e:
        if e.response["Error"]["Code"] == "NoSuchKey":
            if can_log:
                logger.debug(
                    f"Object {s3_key} not found in bucket {aws_s3_bucket_name}."
                )
            return None
        else:
            raise  # Re-raise other exceptions if they occur


def save_json_to_aws_s3_bucket(
    json_to_save: str, s3_key: str, aws_sts_credentials: dict, aws_s3_bucket_name: str
):
    """
    @param json_to_save: e.g. json.dumps(list_of_dicts)
    """
    s3 = boto3.client(
        "s3",
        aws_access_key_id=aws_sts_credentials["AccessKeyId"],
        aws_secret_access_key=aws_sts_credentials["SecretAccessKey"],
        aws_session_token=aws_sts_credentials["SessionToken"],
    )

    logger.debug(
        f"bucket={aws_s3_bucket_name}; key={s3_key}; Attempting json save in aws s3 bucket..."
    )
    s3_key_exists_already = None
    try:
        s3.head_object(Bucket=aws_s3_bucket_name, Key=s3_key)
        s3_key_exists_already = True
    except ClientError as e:
        if e.response["Error"]["Code"] == "404":
            s3_key_exists_already = False
        else:
            raise

    if s3_key_exists_already:  # exit early if object already found
        logger.debug(
            f"bucket={aws_s3_bucket_name}; key={s3_key}; Found key already exists in bucket. Nothing to do."
        )
        return False

    logger.debug(
        f"bucket={aws_s3_bucket_name}; key={s3_key}; Key not found to exist already. Preparing write..."
    )
    s3_write_response = s3.put_object(
        Bucket=aws_s3_bucket_name,
        Key=s3_key,
        Body=json_to_save,
        ContentType="application/json",
    )
    logger.debug(
        f'bucket={aws_s3_bucket_name}; key={s3_key}; version_id={s3_write_response["VersionId"]}; Write done.'
    )

    return True
